/*****
 * xannotate.c
 * Annotate desktop with scribbles, for presentation.
 * Copyright (C) James Budiono, 2014
 * License: GNU GPL Version 3 or later.
 *****/
 
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/extensions/shape.h>
#include <X11/extensions/Xrender.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

//#define USE_SCREENSHOT // enable this to enable screenshot

#ifdef USE_SCREENSHOT
#include "pnglite.h"
#endif

#define DEFAULT_WIDTH   5
#define DEFAULT_ERASER_WIDTH 20
#define DEFAULT_HOTKEY "Pause"
#define DEFAULT_SCREENSHOT_PREFIX "xannotate"
#define DEFAULT_PEN1 "Red"
#define DEFAULT_PEN2 "Green"
#define DEFAULT_PEN3 "Blue"

// so we don't pass thousands of parameters each time
typedef struct
{
	Display *display;
	Window	root_window;
	int     screen;
	int     width, height;
	int     use_alpha;
	
	Window overlay;	
	Pixmap overlay_content, overlay_shapemask;
	XVisualInfo overlay_visinfo;
	Colormap cmap;
	unsigned int hotkey_keycode, hotkey_keycode2;
	
	Atom wm_delete_window;
	Atom wm_protocols;
		
	GC content_pen1, content_pen2, content_pen3, active_pen;
	GC content_blank;
	GC bitmap_white, bitmap_black;
	GC content_eraser, bitmap_eraser;
	GC content_tool, bitmap_tool;
	
	Cursor draw_cursor, eraser_cursor;
	Cursor tool_cursor;

	enum {PEN1=1, PEN2, PEN3, MAXPEN} pen_index;
	int mouse_grabbed, visible;
	enum {PEN, ERASER} tool_type;
	int do_draw, lastx, lasty;
	char *screenshot_path;
	
	// xrender stuff
	Picture overlay_picture, content_picture;
} ProgState;

// This must be freed when done
char *get_save_fname(ProgState *st)
{
	char *fname;
	char ts[64];
	
	time_t now = time(NULL);
	struct tm *now2 = localtime(&now);
	strftime(ts, sizeof(ts), "%Y%m%dT%H%M%S", now2);
	asprintf (&fname, "%s-%s.png", st->screenshot_path, ts);
	return fname;
}

// create the overlay window we will use as canvas
void create_overlay_window (ProgState *st, Window *parent) 
{
	XSetWindowAttributes xswa;
	unsigned long attrmask=0;

	if (st->use_alpha)
	{
		if (XMatchVisualInfo(st->display, st->screen, 32, TrueColor, &st->overlay_visinfo))
		{
			xswa.colormap = st->cmap = XCreateColormap (
				st->display, *parent, st->overlay_visinfo.visual, AllocNone
			);
			xswa.background_pixel = 0;
			xswa.border_pixel = 0;
			attrmask |= CWColormap|CWBackPixel|CWBorderPixel;
		}
		else
		{
			fprintf (stderr, "Unable to enable alpha, falling back to XSHAPE.\n");
			st->use_alpha = 0;
		}
	}
	
	if (!st->use_alpha)
	{
		st->overlay_visinfo.visual = DefaultVisual (st->display, st->screen);
		st->overlay_visinfo.depth  = DefaultDepth (st->display, st->screen);
		st->cmap = DefaultColormap(st->display, st->screen);
		xswa.background_pixel = WhitePixel(st->display, st->screen);
		attrmask |= CWBackPixel;
	}

	xswa.save_under = True;
	xswa.backing_store = NotUseful;
	xswa.override_redirect = True;
	xswa.cursor = None;
	attrmask |= CWBackingStore|CWSaveUnder|CWOverrideRedirect|CWCursor;
	
	st->overlay = XCreateWindow (st->display, *parent, 0,0, st->width,st->height, 0, 
	                     st->overlay_visinfo.depth, InputOutput, st->overlay_visinfo.visual, 
	                     attrmask, &xswa);
	                     
    XShapeCombineRectangles (st->display, st->overlay, ShapeInput, 0, 0, NULL, 0, ShapeSet, False);
	XSelectInput(st->display, st->overlay, StructureNotifyMask|ExposureMask);
    XSetWMProtocols(st->display, st->overlay, &st->wm_delete_window, 1);
    
	XMapWindow(st->display, st->overlay);
}


// create Picture objects for XRenderComposite operation
void xrender_setup (ProgState *st)
{
	fprintf (stderr, "Using XRENDER composite extension.\n");
	st->overlay_picture = XRenderCreatePicture(
		st->display, st->overlay, XRenderFindStandardFormat(st->display, PictStandardARGB32), 
		0, 0
	);
	st->content_picture = XRenderCreatePicture(
		st->display, st->overlay_content, XRenderFindStandardFormat(st->display, PictStandardARGB32), 
		0, 0
	);
}


// create GC (pen) for drawing, copying, etc
GC create_pen (ProgState *st, Drawable *d, char *colour, int width)
{
	GC pen;
	XGCValues values;
	XColor xc, xc2;	
	
	xc.pixel = 0; // transparent
	if (colour)
		XAllocNamedColor(st->display, st->cmap, colour, &xc, &xc2);
	
	/* create the pen to draw lines with */
	values.line_width = width;
	values.line_style = LineSolid;
	values.foreground = xc.pixel;
	values.cap_style = CapRound;
	values.join_style = JoinMiter;
	return XCreateGC(st->display, *d, GCForeground|GCLineWidth|GCLineStyle|GCCapStyle|GCJoinStyle, &values);
}


// enable notification when our hotkey is pressed
char *grab_hotkey (ProgState *st, char *hotkey)
{
	char *hotkey_name = hotkey;
	unsigned int keycode = 0, keycode2 = 0;
	KeySym keysym, keysym2;
	
	keysym = XStringToKeysym (hotkey);
	if (keysym != NoSymbol)
		keycode = XKeysymToKeycode (st->display, keysym);
	else {
		keycode = atoi(hotkey);
		if (keycode) {
			keysym2 = XKeycodeToKeysym(st->display, keycode, 0);
			keycode2 = XKeysymToKeycode (st->display, keysym2);
			hotkey_name = XKeysymToString (keysym2);
		}
	}
		
	if (keycode)
		XGrabKey (st->display, keycode, AnyModifier, st->root_window, 
		          False, GrabModeAsync, GrabModeAsync);
	if (keycode2 && keycode2 != keycode)
		XGrabKey (st->display, keycode2, AnyModifier, st->root_window, 
		          False, GrabModeAsync, GrabModeAsync);	
	if (!keycode && !keycode2) 
	{
		fprintf (stderr, "I don't understand %s, bailing out.\n", hotkey);
		exit(1);
	}
	st->hotkey_keycode = keycode;
	st->hotkey_keycode2 = keycode2;
	return hotkey_name;
}


// grap or ungrab the mouse pointer
int grab_mouse (ProgState *st, int grab)
{
	if (grab) {
		if (!st->mouse_grabbed && st->visible) {
			fprintf (stderr, "mouse grab\n");
			XGrabPointer (st->display, st->overlay, False, 
			              ButtonPressMask|ButtonReleaseMask|PointerMotionMask,
			              GrabModeAsync, GrabModeAsync, None, None, CurrentTime);
			XDefineCursor(st->display, st->overlay, st->tool_cursor);
			st->mouse_grabbed = grab;
		}
	} else {
		if (st->mouse_grabbed) {
			fprintf (stderr, "mouse release\n");
			XUngrabPointer (st->display, CurrentTime);
			XDefineCursor(st->display, st->overlay, None);
			st->mouse_grabbed = grab;
		}		
	}
}


// show or hide window
void show_window (ProgState *st, int visible)
{
	if (visible) {
		if (!st->visible) {
			fprintf (stderr, "show window\n");
			XMapWindow(st->display, st->overlay);
			st->visible = visible;
		}
	} else {
		if (st->visible) {
			fprintf (stderr, "hide window\n");
			grab_mouse (st, 0);
			XUnmapWindow(st->display, st->overlay);
			st->visible = visible;
		}		
	}
}


// update window content with latest drawing
void repaint (ProgState *st)
{	
	if (st->use_alpha)
	{
		XRenderComposite(st->display, PictOpSrc, st->content_picture, None,
                         st->overlay_picture, 0, 0, 0, 0, 0, 0,
                         st->width, st->height);
	}
	else 
	{
		XCopyArea (st->display, st->overlay_content, st->overlay, st->content_blank, 
				   0, 0, st->width, st->height, 0, 0);		
		XShapeCombineMask(st->display, st->overlay, ShapeBounding, 0, 0, st->overlay_shapemask, ShapeSet);
	}
}


// draw a line
void draw_line (ProgState *st, int to_x, int to_y)
{
	//fprintf (stderr, "draw %d, %d, %d, %d\n", st->lastx, st->lasty, to_x, to_y);
	XDrawLine(st->display, st->overlay_content, st->content_tool, st->lastx, st->lasty, to_x, to_y);
	if (!st->use_alpha)
		XDrawLine(st->display, st->overlay_shapemask, st->bitmap_tool, st->lastx, st->lasty, to_x, to_y);
	repaint(st);
}


// clear canvas
void clear_drawing (ProgState *st)
{
	fprintf (stderr, "clear drawing\n");
	XFillRectangle (st->display, st->overlay_content, st->content_blank, 0, 0, st->width, st->height);
	XFillRectangle (st->display, st->overlay_shapemask, st->bitmap_black, 0, 0, st->width, st->height);	
}


#ifdef USE_SCREENSHOT
// get shift count based on mask, assumes LSB
int get_shift (int mask) {
	int n=0, test=1;
	while (n < sizeof(mask)*8) {
		if (mask & test) return n;
		n++; test<<=1;
	}
	return 0;
}

// get 24-bit rgb data from XImage, ZPixmap format
// only supports direct/truecolor 15/16/24 bit, assumes LSB
unsigned char *extract_rgb24_from_ximage (XImage *image) {
	unsigned char *out=NULL;
	int x,y,index;
	int width, height;
	unsigned long pix;
	
	int red_shift, green_shift, blue_shift;
	int red_mask, green_mask, blue_mask;
	double red_factor=1.0, green_factor=1.0, blue_factor=1.0;
	
	width       = image->width;
	height      = image->height;
	red_mask    = image->red_mask;
	green_mask  = image->green_mask;
	blue_mask   = image->blue_mask;
	red_shift   = get_shift (red_mask);
	green_shift = get_shift (green_mask);
	blue_shift  = get_shift (blue_mask);
	
	switch (image->depth) {
		case 15:
		case 16:
			red_factor   = 255.0/(red_mask   >> red_shift);
			green_factor = 255.0/(green_mask >> green_shift);
			blue_factor  = 255.0/(blue_mask  >> blue_shift);		
		case 24:
		case 32:
			break;
		default:
			fprintf (stderr, "Unsupported bit depth: %d.\n", image->depth);
			return NULL;
	}
	
	out = malloc (width * height * 3); // 24-bit RGB (3 bytes per pixel)	
	for (index=0, y=0; y < height; y++)
	{
		for (x=0; x < width; x++)
		{
			pix = XGetPixel (image, x,y);
			out[index++] = ((pix & red_mask)   >> red_shift) * red_factor;
			out[index++] = ((pix & green_mask) >> green_shift) * green_factor;
			out[index++] = ((pix & blue_mask)  >> blue_shift) * blue_factor;
		}
	}
	return out;
}

// take screenshot (if enabled), using pnglite
void take_screenshot (ProgState *st)
{
	XImage *image;
	png_t png;
	unsigned char *buf;
	char *fname=NULL;
	
	// do actual screenshot
	image = XGetImage (st->display, st->root_window, 0,0, st->width, st->height, AllPlanes, ZPixmap);
	buf = extract_rgb24_from_ximage (image);
	
	// save it
	if (buf) {
		fname = get_save_fname(st);
		png_open_file_write (&png, fname);
		png_set_data (&png, image->width, image->height, 8, PNG_TRUECOLOR, buf); //8 bits per channel.
		png_close_file (&png);
		free (buf);	
		free (fname);
	}
	XDestroyImage (image);
	fprintf (stderr, "screenshot\n");	
}
#endif


// set active pen
set_active_pen (ProgState *st, int pen_index)
{
	st->pen_index = pen_index;
	if (st->pen_index >= MAXPEN) st->pen_index=PEN1;
	switch (st->pen_index) 
	{
		case PEN1: st->active_pen = st->content_pen1; break;
		case PEN2: st->active_pen = st->content_pen2; break;
		case PEN3: st->active_pen = st->content_pen3; break;
	}
}


// toggle between pen and eraser
set_tool (ProgState *st, int type)
{
	switch (type) 
	{
		case PEN:
			fprintf (stderr, "pen%d\n",st->pen_index);
			st->tool_type = type;			
			st->content_tool = st->active_pen;
			st->bitmap_tool = st->bitmap_white;
			st->tool_cursor = st->draw_cursor;
			break;

		case ERASER:
			fprintf (stderr, "eraser\n");
			st->tool_type = type;
			st->content_tool = st->content_eraser;
			st->bitmap_tool = st->bitmap_eraser;
			st->tool_cursor = st->eraser_cursor;
			break;
	}
	XDefineCursor(st->display, st->overlay, st->tool_cursor);
}


// main event loop
int event_loop (ProgState *st)
{
	XEvent ev;
	int running=1;
	int retcode=0;
	int state;	
	
	/* as each event that we asked about occurs, we respond. */
	while(running)
	{
next_event:
		XNextEvent(st->display, &ev);
		
		switch(ev.type)
		{
			case ButtonPress:
				// bare click 
				if (!(ev.xbutton.state & ShiftMask))
				{
					switch (ev.xbutton.button)
					{
						case Button1: // draw with pen
							if (st->tool_type != PEN) set_tool (st, PEN);
							break;
						case Button3: // draw with eraser
							if (st->tool_type != ERASER) set_tool (st, ERASER);
							break;
						default: // ignore the rest
							goto next_event;
					}
				} else {
					// shift-left-click - change pen
					if (ev.xbutton.button == Button1) 
					{
						set_active_pen (st, ++st->pen_index);
						set_tool (st, PEN);
					}
					goto next_event;
				}
				st->do_draw = 1;
				st->lastx = ev.xbutton.x;
				st->lasty = ev.xbutton.y;
				//fprintf(stderr, "Press %d, %d, %d, %d, %d\n",ev.xbutton.window, ev.xbutton.x,ev.xbutton.y,ev.xbutton.x_root,ev.xbutton.y_root);
				break;

			case ButtonRelease:
				// only consider left/right click
				if (!ev.xbutton.button == Button1 && !ev.xbutton.button == Button3) goto next_event;
				if (st->do_draw) draw_line (st, ev.xmotion.x, ev.xmotion.y);
				st->do_draw = 0;
				if (st->tool_type != PEN) set_tool (st, PEN); // always reset tool to pen
				//fprintf(stderr, "Release %d, %d, %d, %d, %d\n",ev.xbutton.window, ev.xbutton.x,ev.xbutton.y,ev.xbutton.x_root,ev.xbutton.y_root);
				break;
			
			case MotionNotify:
				if (st->do_draw) {
					draw_line (st, ev.xmotion.x, ev.xmotion.y);
					st->lastx = ev.xmotion.x;
					st->lasty = ev.xmotion.y;
				}
				//fprintf(stderr, "Move %d, %d, %d, %d, %d\n",ev.xmotion.window, ev.xmotion.x,ev.xmotion.y,ev.xmotion.x_root,ev.xmotion.y_root);
				break;
				
			case Expose:
				repaint(st);
				break;

			case ConfigureNotify:
				//todo - if needed
				break;

			case KeymapNotify:
				XRefreshKeyboardMapping(&ev.xmapping);
				break;
				
			case KeyRelease:
				break; //ignore
				
			case KeyPress:
				if (ev.xkey.keycode == st->hotkey_keycode ||
				    ev.xkey.keycode == st->hotkey_keycode2 )
				{
					state = 0;
					if (ev.xkey.state & ShiftMask) state ^= 1;
					if (ev.xkey.state & ControlMask) state ^= 2;
					if (ev.xkey.state & Mod1Mask) state ^= 4; // Alt
					switch (state)
					{
						case 0: // bare - mouse grab
							//fprintf(stderr, "Hotkey pressed, toggling mouse grab.\n");
							grab_mouse (st, !st->mouse_grabbed);
							break;
						case 1: // shift - show/hide
							//fprintf(stderr, "Shift-Hotkey pressed, toggle visibility.\n");
							show_window (st, !st->visible);
							break;
						case 2: // ctrl - clear
							//fprintf(stderr, "Ctrl-Hotkey pressed, clearing.\n");
							if (st->visible) {
								clear_drawing(st);
								repaint(st);
							}
							break;
						case 3: // ctrl-shift - exit 
							//fprintf(stderr, "Ctrl-Shift-Hotkey pressed, exiting.\n");
							XCloseDisplay(st->display);
							running=0;
							break;
						case 4: // alt - unused
							//fprintf(stderr, "Alt-Hotkey pressed.\n");
							break;
						case 5: // alt-shift - screenshot
#ifdef USE_SCREENSHOT
							take_screenshot(st);
#else
							fprintf (stderr, "screenshot feature not enabled.\n");
#endif
							break;
					}
				}				
				break;
				
			case ClientMessage:
				if (ev.xclient.message_type == st->wm_protocols && 
				   (Atom)ev.xclient.data.l[0] == st->wm_delete_window)
				   running=0;
				   break;
		}
	}
	return retcode;
}


//***************** Main ******************
int main (int argc, char*argv[])
{
	ProgState st;
	char *hotkey  = DEFAULT_HOTKEY;
	int pen_width = DEFAULT_WIDTH;
	int eraser_width = DEFAULT_ERASER_WIDTH;
	char *pen1=DEFAULT_PEN1, *pen2=DEFAULT_PEN2, *pen3=DEFAULT_PEN3;
	int opt;

	// argument processing
	st.use_alpha = 0;	
	asprintf (&st.screenshot_path, "%s/" DEFAULT_SCREENSHOT_PREFIX, getenv("HOME"));
	while ((opt = getopt(argc, argv, "1:2:3:n:p:k:w:ha")) != -1) 
	{
		switch (opt) 
		{
			case 'k':
				if (optarg) hotkey = optarg;
				break;
			case 'w':
				pen_width = atoi (optarg);
				if (!pen_width) pen_width = DEFAULT_WIDTH;
				break;
			case 'e':
				eraser_width = atoi (optarg);
				if (!eraser_width) eraser_width = DEFAULT_ERASER_WIDTH;
				break;
			case 'p':
				if (optarg) st.screenshot_path = optarg;
				break;
			case '1':
				if (optarg) pen1 = optarg;
				break;
			case '2':
				if (optarg) pen2 = optarg;
				break;			
			case '3':
				if (optarg) pen3 = optarg;
				break;
			case 'a':
				st.use_alpha = 1;
				break;
			case 'h':
				fprintf (stderr,
"Usage: xannotate [-k key] [-w width] [-e width] [-p path] [-n index]\n"
"                 [-123 colour]\n"
"Copyright (C) James Budiono, 2014. License: GPL version 3.0 or later\n\n"
" -h       : this text\n"
" -k key   : use 'key' as the hotkey (default is 'Pause')\n"
" -w width : use 'width' as the default pen width (default is 5)\n"
" -e width : use 'width' as the default eraser width (default is 20)\n"
" -p path  : path/filename for screenshot (default is $HOME/" DEFAULT_SCREENSHOT_PREFIX ")\n"
" -a       : enable alpha for use with XRENDER composite. You must be running\n"
"            an X composite manager (xcompmgr, compton, kwin) for this to work.\n"
" -1 colour: colour for pen1 (default red)\n"
" -2 colour: colour for pen2 (default green)\n"
" -3 colour: colour for pen2 (default blue)\n"
);
				return 0;
		}
	}

	// initialisations
#ifdef USE_SCREENSHOT	
	png_init (0,0);
#endif
	// startup defaults
	st.mouse_grabbed = 0;
	st.do_draw = 0;
	st.visible = 1;	
	
    st.display = XOpenDisplay(NULL);
    if (st.display == NULL) {
        fprintf(stderr, "Cannot open display\n");
        exit(1);
    }
    st.screen            = DefaultScreen (st.display);
    st.root_window       = DefaultRootWindow (st.display);
    st.width             = DisplayWidth(st.display, st.screen);
    st.height            = DisplayHeight(st.display, st.screen);
    st.wm_delete_window  = XInternAtom(st.display, "WM_DELETE_WINDOW", False);
    st.wm_protocols      = XInternAtom(st.display, "WM_PROTOCOLS", False);
    st.hotkey_keycode    = st.hotkey_keycode2 = 0;
    char *hotkey_name    = grab_hotkey (&st, hotkey);
    
	fprintf (stderr,
"--- Key usage ---\n"
"%s            - toggle draw/cursor mode\n"
"Shift-%s      - show/hide drawing\n"
"Ctrl-%s       - clear drawing\n"
"Ctrl-Shift-%s - exit\n"
"Alt-Shift-%s  - screenshot\n"
"--- Mouse usage ---\n"
"Left-click  - pen: press to start, move to draw, release to stop\n"
"Right-click - eraser: press to start, move to erase, release to stop\n"
"Shift-Left-click    - switch pen\n"
"---\n", hotkey_name, hotkey_name, hotkey_name, hotkey_name, hotkey_name, hotkey_name);    

    create_overlay_window (&st, &st.root_window);
    st.overlay_content   = XCreatePixmap (st.display, st.overlay, st.width, st.height, st.overlay_visinfo.depth);
    st.overlay_shapemask = XCreatePixmap (st.display, st.overlay, st.width, st.height, 1);

	st.draw_cursor       = XCreateFontCursor (st.display, XC_pencil); 
	st.eraser_cursor     = XCreateFontCursor (st.display, XC_icon); 
    
    st.content_pen1      = create_pen (&st, &st.overlay_content, pen1, pen_width);
    st.content_pen2      = create_pen (&st, &st.overlay_content, pen2, pen_width);
    st.content_pen3      = create_pen (&st, &st.overlay_content, pen3, pen_width);
    st.content_blank     = create_pen (&st, &st.overlay_content, 0, pen_width);
    st.bitmap_black      = create_pen (&st, &st.overlay_shapemask, "Black", pen_width);
    st.bitmap_white      = create_pen (&st, &st.overlay_shapemask, "White", pen_width);
    
    st.content_eraser    = create_pen (&st, &st.overlay_content, 0, eraser_width);
    st.bitmap_eraser     = create_pen (&st, &st.overlay_shapemask, "Black", eraser_width);

	st.tool_type         = PEN;
	st.pen_index         = PEN1;
    st.content_tool      = st.active_pen = st.content_pen1;
    st.bitmap_tool       = st.bitmap_white;
	st.tool_cursor       = st.draw_cursor;     

    clear_drawing (&st);
    if (st.use_alpha) xrender_setup (&st);
    repaint (&st);    
    return event_loop (&st);
}

