/******************
 * Geocode IP addresses
 * Uses ipinfo2.dat prepared by 'preprocess2'
 * Usage: geocode2 ipinfo2.dat 
 * Input: stdin ip addresses, one per line
 * Output: stdout, location code, one per line
 * Note: IPv4 only
 * (C) James Budiono 2013, 2019
 * License: GNU GPL Version 3 or later
 * ****************/
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <stdint.h>
#include <string.h>

//#define DEBUG
#include "ipgeocode2.h"

//replacement for inet_network
uint32_t convert_to_ip_address (char *s) {
	uint64_t ip_address = 0;
	char * next = 0;
	int i;
	
	for (i=0; i<4; i++) {
		ip_address += (uint32_t) strtoul (s, &next, 10);
		ip_address <<= 8;
		if (s != next) s = next + 1;
		else {
			ip_address >>= 8; //undo
			break;
		}
	}
	ip_address >>= 8; //undo off-by-one
	return ip_address << 8*(4-i);
}

void print_error (char *s) {
	write (2, s, strlen(s));
}

int main(int argc, char *argv[]) {
	char inputbuf[256]; //for fgets
	struct_ipblock *ipblock;
	
	int fd_ipinfo;
	struct stat sb_ipinfo;
	uint32_t ipblock_len, low, high, test;
	uint32_t ip_address;

	// open files
	if (argc < 2) { print_error ("usage: geocode2 ipinfo2.dat\n"); exit(1); }
	fd_ipinfo = open(argv[1], O_RDONLY);
	if (fd_ipinfo == -1) handle_error("open ipinfo2");

	if (fstat(fd_ipinfo, &sb_ipinfo) == -1) handle_error("fstat ipinfo2");
	ipblock = mmap(NULL, sb_ipinfo.st_size, PROT_READ, MAP_PRIVATE, fd_ipinfo, 0);
	if (ipblock == MAP_FAILED) handle_error("mmap ipinfo2");
   
	ipblock_len = sb_ipinfo.st_size / sizeof (struct_ipblock);
	
	while (fgets (inputbuf, sizeof(inputbuf), stdin)) {
		// 0. calc ipv4 address
		//ip = inet_network (inputbuf);
		ip_address = convert_to_ip_address (inputbuf);
		debug_print ("searching for %s %u\n", inputbuf, ip_address);

		// 1. find the location
		// 1.1 initial test - make sure we're in range
		test = low = 0; high = ipblock_len-1;
		if (ip_address < ipblock[low].ipstart || ip_address > ipblock[high].ipend) 
			continue;

		// 1.2 non-recursive binary search
		while (low <= high) {
			test = ( low + high ) >> 1;
			debug_print ("%u %u %u\n", low, test, high);
			if (ip_address >= ipblock[test].ipstart && ip_address <= ipblock[test].ipend) break;
			else if (ip_address < ipblock[test].ipstart) high = test - 1;
			else low = test + 1;
		}
		printf("%.4f %.4f\n", ipblock[test].lat, ipblock[test].lng);
	}
	
	// finish & cleanup
	munmap (ipblock, sb_ipinfo.st_size); 
	close (fd_ipinfo);
	exit(0);	
}
