/******************
 * Geocode IP addresses
 * Uses ipinfo.dat and locinfo.dat prepared by 'preprocess'
 * Usage: geocode ipinfo.dat locinfo.dat
 * Input: stdin ip addresses, one per line
 * Output: stdout, location code, one per line
 * Note: IPv4 only
 * (C) James Budiono 2013
 * License: GNU GPL Version 3 or later
 * ****************/
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <stdint.h>
#include <string.h>

#include "ipgeocode.h"

//replacement for inet_network
uint32_t convert_to_ip_address (char *s) {
	uint64_t ip_address = 0;
	char * next = 0;
	int i;
	
	for (i=0; i<4; i++) {
		ip_address += (uint32_t) strtoul (s, &next, 10);
		ip_address <<= 8;
		if (s != next) s = next + 1;
		else {
			ip_address >>= 8; //undo
			break;
		}
	}
	ip_address >>= 8; //undo off-by-one
	return ip_address << 8*(4-i);
}

void print_error (char *s) {
	write (2, s, strlen(s));
}

int main(int argc, char *argv[]) {
	char inputbuf[256]; //for fgets
	struct_ipblock *ipblock;
	struct_location *loc;
	
	int fd_ipinfo, fd_locinfo;
	struct stat sb_ipinfo, sb_locinfo;
	uint32_t ipblock_len, low, high, test;
	uint32_t ip_address, ip_location, max_location;

	// open files
	if (argc < 3) { print_error ("usage: geocode ipinfo.dat locinfo.dat\n"); exit(1); }
	fd_ipinfo = open(argv[1], O_RDONLY);
	if (fd_ipinfo == -1) handle_error("open ipinfo");
	fd_locinfo = open(argv[2], O_RDONLY);
	if (fd_ipinfo == -1) handle_error("open locinfo");

	if (fstat(fd_ipinfo, &sb_ipinfo) == -1) handle_error("fstat ipinfo");
	ipblock = mmap(NULL, sb_ipinfo.st_size, PROT_READ, MAP_PRIVATE, fd_ipinfo, 0);
	if (ipblock == MAP_FAILED) handle_error("mmap ipblock");

	if (fstat(fd_locinfo, &sb_locinfo) == -1) handle_error("fstat locinfo");
	loc = mmap(NULL, sb_locinfo.st_size, PROT_READ, MAP_PRIVATE, fd_locinfo, 0);
	if (loc == MAP_FAILED) handle_error("mmap ipblock");
   
	ipblock_len = sb_ipinfo.st_size / sizeof (struct_ipblock);
	max_location = sb_locinfo.st_size / sizeof (struct_location);
	
	while (fgets (inputbuf, sizeof(inputbuf), stdin)) {
		// 0. calc ipv4 address
		//ip = inet_network (inputbuf);
		ip_address = convert_to_ip_address (inputbuf);
		debug_print ("searching for %s %u\n", inputbuf, ip_address);

		// 1. find the location
		// 1.1 initial test - make sure we're in range
		test = low = 0; high = ipblock_len-1;
		if (ip_address < ipblock[low].ipstart || ip_address > ipblock[high].ipend) 
			continue;

		// 1.2 non-recursive binary search
		while (low <= high) {
			test = ( low + high ) >> 1;
			debug_print ("%u %u %u\n", low, test, high);
			if (ip_address >= ipblock[test].ipstart && ip_address <= ipblock[test].ipend) break;
			else if (ip_address < ipblock[test].ipstart) high = test - 1;
			else low = test + 1;
		}
		ip_location = ipblock[test].location - 1; //array starts from zero
		
		//2. print the latitude / longitude
		debug_print ("looking up index %u\n", ip_location);
		if (ip_location < max_location)
			write (1, loc[ip_location].latlong, loc[ip_location].len);
	}
	
	// finish & cleanup
	munmap (ipblock, sb_ipinfo.st_size); 
	munmap (loc, sb_locinfo.st_size); 
	close (fd_ipinfo);
	close (fd_locinfo);
	exit(0);	
}
